/*
Program : Strip (Secure Tool for Recalling Important Passwords) 
Description: A secure password and account manager for the Palm(t) Computing Platform 
Copyright (C) 1999  Stephen J Lombardo

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

Strip has been written and developed by Stephen J Lombardo (Zetetic Enterprises) 1999

Contact Info:
lombardos@zetetic.net
http://www.zetetic.net/
Zetetic Enterprises
348 Wasington Ave 
Clifton NJ, 07011

Bug reports and feature requests should be sent to bugs@zetetic.net.


------RSA Data Security, Inc. MD5 Message Digest Algorithm-------------
Strip uses the MD5 message digest algorithm, Copyright (C) 1990, 
RSA Data Security, Inc. All rights reserved. See md5.c or md5.h 
for specific terms and warranty disclaimer from RSA Data Security Inc.
-----------------------------------------------------------------------

------Idea block encryption---------------------------------------
Strip uses the Idea block encryption algoritm, Copyright (C) Ascom. All rights reserved.
Idea is a patented algorithm. It is free for non commercial use, if you wish to use 
this product or the Idea algorithm in general for commercial purposes, you must purchase a license
from Ascom at http://www.ascom.ch/infosec/idea/pricing.html
-----------------------------------------------------------------------

        <**  DO NOT EXPORT **>
Strip uses strong cryptography. "Idea" is a block algoritm with a
128 bit key length, and "MD5" creates a 128 bit message digest. In the 
United states it is currently illegal to export products describing
or incorporating encryption techniques with key lengths greater
than 40 bits. It is therefore illegal to export this program in
any format. Please dont get the government on my back...
*/

#include <Pilot.h>          
#include <ExgMgr.h>
#include "StripRsc.h"
#include "Strip.h"

#ifdef __GNUC__
#include "Callbacks.h"
#endif      


/* Global Variables */
Word currentForm, oldForm;
static DmOpenRef SystemDB, AccountDB, PasswordDB;
Boolean firstRun=false, passwordEchoOff=false, hideSecretRecords, closing=false;
UInt currentSID=0, currentAID=0, selectedAccount=0, selectedSystem=0;
char * SysPass;
static  char * emptyString="\0";
System CurSys;
Account CurAcc;
RecBuff CurSysScratch, CurAccScratch, CurComment;
ULong lastEventTime, autoOffTime, oldAutoOffTime;


/************************************************************************
 * Function: GetObjectFromActiveForm
 * Description: pass this function the object id of any UI object in the 
 * currently active form and it will return a valid pointer to the 
 * UI object.
 * *********************************************************************/
static VoidPtr GetObjectFromActiveForm(Word objectID)
{
    FormPtr curForm=FrmGetActiveForm();
    return FrmGetObjectPtr(curForm, FrmGetObjectIndex(curForm, objectID));
}

/***********************************************************************
 * Function: SortPosAccountFunction
 * Description: Callback function to find the proper insertion index 
 * for an account based on alphabetical order of the account username
 * *********************************************************************/
static Int SortPosAccountFunction(Account *rec1, RecordBuffer rec2, Int unused, SortRecordInfoPtr unused1, 
        SortRecordInfoPtr unused2, VoidHand appInfo)
{
    Int result, sysid;
    Account ac;
    RecordBuffer scratch;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif

        // pop the system id off the record.  If the sysid is -1 then the record has been beamed
        // and is not encrypted, therefore we always return 1 so the unencrypted record ends
        // up at the beginning of the database.
    MemMove(&sysid, rec2, sizeof(sysid));
    if(sysid==-1)
        result=1;
    else
    {
        if(scratch=MemPtrNew(MemPtrSize(rec2)))
        {
                // unpack the second account and compare the usernames
            UnpackAccount(&ac, rec2, scratch, SysPass, MemPtrSize(rec2), true, true);
            result=StrCompare(rec1->username, ac.username);
            MemSet(scratch, MemPtrSize(scratch), 0);
            MemPtrFree(scratch);
        }
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return result;
}

/*static Int CompareAccountFunction( RecordBuffer rec1, RecordBuffer rec2, Int unused, SortRecordInfoPtr unused1, 
        SortRecordInfoPtr unused2, VoidHand appInfo)
{
    Int result;
    Account one, two;
    RecordBuffer scratch1, scratch2;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
        
    if(scratch1=MemPtrNew(MemPtrSize(rec1))&&
        scratch2=MemPtrNew(MemPtrSize(rec2)))
    {

        UnpackAccount(&one, rec1, scratch1, SysPass, MemPtrSize(rec1), true, true);
        UnpackAccount(&two, rec2, scratch2, SysPass, MemPtrSize(rec2), true, true);
        
        result=StrCompare(one.username, two.username);
        MemSet(scratch1, MemPtrSize(scratch1), 0);
        MemSet(scratch2, MemPtrSize(scratch2), 0);
        MemPtrFree(scratch1);
        MemPtrFree(scratch2);
    }
    
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return result;
}*/

/***********************************************************************
 * Function: SortPosSystemFunction
 * Description: Callback function to find the proper insertion index 
 * for an system based on alphabetical order of the system name
 * *********************************************************************/
static Int SortPosSystemFunction(System *rec1, RecordBuffer rec2, Int unused, SortRecordInfoPtr unused1, 
        SortRecordInfoPtr unused2, VoidHand appInfo)
{
    Int result, index;
    System sys;
    RecordBuffer scratch;

#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    if(scratch=MemPtrNew(MemPtrSize(rec2)))
    {
                // unpack the second system and compare the names
        UnpackSystem(&sys, rec2, scratch, SysPass, MemPtrSize(rec2), true);
        result=StrCompare(rec1->name, sys.name);
        MemSet(scratch, MemPtrSize(scratch), 0);
        MemPtrFree(scratch);
    }

#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return result;
}

/*static Int CompareSystemFunction(RecordBuffer rec1, RecordBuffer rec2, Int unused, SortRecordInfoPtr unused1, 
        SortRecordInfoPtr unused2, VoidHand appInfo)
{
    Int result;
    System one, two;
    RecordBuffer scratch1, scratch2;

#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
        
    if(scratch1=MemPtrNew(MemPtrSize(rec1))&&
        scratch2=MemPtrNew(MemPtrSize(rec2)))
    {
        UnpackSystem(&one, rec1, scratch1, SysPass, MemPtrSize(rec1), true);
        UnpackSystem(&two, rec2, scratch2, SysPass, MemPtrSize(rec2), true);
        result=StrCompare(one.name, two.name);
        MemSet(scratch1, MemPtrSize(scratch1), 0);
        MemSet(scratch2, MemPtrSize(scratch2), 0);
        MemPtrFree(scratch1);
        MemPtrFree(scratch2);
    }

#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif

    return result;
}*/


/************************************************************************
 * Function: setFieldFromHandle
 * Description:  Pass the function the object id of a field and a handle
 * containing the text and it sets the field value to the text handle.
 * It will return a pointer to the set field.
 * **********************************************************************/
static FieldPtr setFieldFromHandle(Word field, VoidHand text)
{
    VoidHand oldText;
    FieldPtr fld;
    
    fld=GetObjectFromActiveForm(field);

        // get a pointer to the old handle
    oldText=(VoidHand)FldGetTextHandle(fld);
    FldSetTextHandle(fld, (Handle)text);
    FldDrawField(fld);

        // deallocate the old handle 
    if(oldText)
    {
        VoidPtr p=MemHandleLock(oldText);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(oldText);
        MemHandleFree(oldText);
    }
    
    return fld;
}

/************************************************************************
 * Function: setFieldFromString
 * Description:  Pass the function the object id of a field and a string
 * containing the text and it creates a new handle, and calls setField from
 * handle to set the field value. 
 * It will return a pointer to the set field.
 * **********************************************************************/
static FieldPtr setFieldFromString(Word field, CharPtr str)
{
    VoidHand text;

    text=MemHandleNew(StrLen(str)+1);
    if(!text)
        return NULL;

        // copy the string to the new handle
    StrCopy(MemHandleLock(text), str);
    MemHandleUnlock(text);
    return setFieldFromHandle(field, text);
}

/**************************************************************************
 * Function: SetAttributes
 * Description: pass it a set of variables representing attributes of a field
 * including editablility, underlined, scrollable, and visible. Also pass
 * it the object id of the field you wish to set for.
 * ************************************************************************/
static void SetAttributes(Word edit, Word under, Word scroll, Boolean visible, Word fieldID)
{
    FieldPtr fld;
    FieldAttrType attr;

    fld=GetObjectFromActiveForm(fieldID);

        // get the current attributes, change some of them.
    FldGetAttributes(fld , &attr);
    attr.editable=edit;
    attr.underlined=under;
    attr.hasScrollBar=scroll;

        // set the new attributes.
    FldSetAttributes(fld, &attr);
    
        // if not visible erase the field.
    if(!visible)
        FldEraseField(fld);
}

/************************************************************************
 * Function: freeCache
 * Description: free the global password, system, and account cache.
 * **********************************************************************/
static void freeCache()
{
    VoidPtr p;
    if(SysPass)
    {
        MemSet(SysPass, MemPtrSize(SysPass), 0);
        MemPtrFree(SysPass);
    }
            
    if(CurSysScratch)
    {
        p=MemHandleLock(CurSysScratch);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(CurSysScratch);
        MemHandleFree(CurSysScratch);
    }
    if(CurAccScratch)
    {
        p=MemHandleLock(CurAccScratch);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(CurAccScratch);
        MemHandleFree(CurAccScratch);
    }
}

/************************************************************************
 * Function: cacheSystem
 * Description: this function will bring a system into the global cache
 * based upon its index withing the system database.
 * **********************************************************************/
static void cacheSystem(UInt index)
{
        // if there is already a cached system, free it.
    if(CurSysScratch)
    {
        VoidPtr p=MemHandleLock(CurSysScratch);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(CurSysScratch);
        MemHandleFree(CurSysScratch);
    }
        // get the new system.
    if(CurSysScratch=MemHandleNew(1))
        getSystemFromIndex(index, CurSysScratch, &CurSys);
       
}

/***********************************************************************
 * Function: cacheAccount
 * Description: this function will bring an account into the global cache
 * based upon its index withing the account database.
 * *********************************************************************/
static void cacheAccount(UInt index)
{
        // if there is already a cached account, free it.
    if(CurAccScratch)
    {
        VoidPtr p=MemHandleLock(CurAccScratch);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(CurAccScratch);
        MemHandleFree(CurAccScratch);
    }
        // get the new account
    if(CurAccScratch=MemHandleNew(1))
        getAccountFromIndex(index, CurAccScratch, &CurAcc);
}

/************************************************************************
 * Function: UpdateScrollbar
 * Description: updates the scrollbar for a scrollable field
 * **********************************************************************/
static void UpdateScrollbar(Word fieldID, Word scrollbarID)
{
    
    ScrollBarPtr scroll;
    FieldPtr field;
    Word currentPos;
    Word textHeight;
    Word fieldHeight;
    Word maxValue;
    
        // get a pointer to the field and the scroll values
    field = GetObjectFromActiveForm(fieldID);
    FldGetScrollValues(field, &currentPos, &textHeight, &fieldHeight);

        // calculate max value based on field height and text height
    if(textHeight>fieldHeight)
        maxValue=textHeight-fieldHeight;
    else if(currentPos)
        maxValue=currentPos;
    else
        maxValue=0;

        // get a pointer to the scrollbar and setup the scroll bar.
    scroll = GetObjectFromActiveForm(scrollbarID);
    SclSetScrollBar(scroll, currentPos, 0, maxValue, fieldHeight-1);
}

/***********************************************************************
 * Function: ScrollLines
 * Description: scrolls a field based a variable number of lines and optionally
 * redraws
 * *********************************************************************/
static void ScrollLines(int toScroll, Boolean redraw, Word fieldID, Word scrollbarID)
{
    FieldPtr field;
    
    field = GetObjectFromActiveForm(fieldID);

        // scroll the field according to the number of lines.
    if(toScroll<0)
        FldScrollField(field, -toScroll, up);
    else
        FldScrollField(field, toScroll, down);

        // redraw if neccessary or if requested
    if((FldGetNumberOfBlankLines(field) && toScroll<0 ) || redraw)
        UpdateScrollbar(fieldID, scrollbarID);
}

/*********************************************************************
 * Function: PageScroll
 * Description: Scroll a field up or down one entire page
 * *******************************************************************/
static void PageScroll(DirectionType direction, Word fieldID, Word scrollbarID)
{
    FieldPtr field;

    field=GetObjectFromActiveForm(fieldID);

        // make sure the field is scrollable
    if(FldScrollable(field, direction))
    {
            // find out how many lines are visible
        int toScroll=FldGetVisibleLines(field) -1;

        if(direction==up)
            toScroll = -toScroll;

            // scroll em
        ScrollLines(toScroll, true, fieldID, scrollbarID);
    }
}


/**********************************************************************
 * Function: getDatabase
 * Description: pass the function the necessare database information,
 * and it will either open an existing database or create a new one if 
 * neccessary. "created" will be true if a new database was created
 * *******************************************************************/
static Err getDatabase(DmOpenRef *DBptr, ULong type, ULong creator, ULong mode, UInt card, char *name, Boolean *created)
{
    Err errors;
    LocalID id;
    UInt cardNum, attr;
  
    *created=false;
    *DBptr=DmOpenDatabaseByTypeCreator(type, creator, mode);
    errors = DmGetLastErr();
        
        // if the database does not exist, make a new one.
    if(! *DBptr)
    {
        errors=DmCreateDatabase(0, name, creator, type, false);
        if (errors)
            return errors;
        *created=true;
        *DBptr=DmOpenDatabaseByTypeCreator(type, creator, mode);
        if( ! *DBptr)
            return DmGetLastErr();
    }
    
    DmOpenDatabaseInfo(*DBptr, &id, NULL, NULL, &cardNum, NULL);
    DmDatabaseInfo(cardNum, id, NULL, &attr, NULL, 
        NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    attr |= dmHdrAttrBackup;
    DmSetDatabaseInfo(cardNum, id, NULL, &attr, NULL, 
        NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
}

/********************************************************************
 * Function: addSystem
 * Description:  function responsible for writing a packed System record
 * to the database. 
 * ******************************************************************/
static void addSystem(RecordBuffer sys, VoidHand systemDBrec)
{
    UInt length=0;
    CharPtr ch;
    UInt offset=0;

        // get length of the system buffer
    length=MemPtrSize(sys);
        
        // re-size and write.
    if(MemHandleResize(systemDBrec, length)==0)
    {
        ch=MemHandleLock(systemDBrec);
        DmWrite(ch, offset, sys, length);
        MemHandleUnlock(systemDBrec);
    }
}

/********************************************************************
 * Function: addPassword
 * Description:  function responsible for writing a packed password record
 * to the database. 
 * ******************************************************************/
static void addPassword(RecordBuffer pass, VoidHand pwDBrec)
{
    UInt length=0;
    CharPtr ch;
    UInt offset=0;

    length= 24;

        // resize the handle and write.
    if(MemHandleResize(pwDBrec, length)==0)
    {
        ch=MemHandleLock(pwDBrec);
        DmWrite(ch, offset,  pass, length);
        MemHandleUnlock(pwDBrec);
    }
}

/*********************************************************************
 * Function: getSystemSize
 * Description: pass the function a pointer to a system and it 
 * will return the buffer length needed to hold that system when
 * it is encrypted. 
 * Note: uses getSCSize function included from twDriver.c
 * *******************************************************************/
static UInt getSystemSize(System *sys)
{
    UInt length=0;
    length= sizeof(sys->SystemID)+StrLen(sys->name)+1;
    return getSCSize(length);
}

/*********************************************************************
 * Function: getAccountSize
 * Description: pass the function a pointer to an account and a boolean
 * for whether or not the account will be encrypted and it
 * will return the buffer length needed to hold that account. 
 * Note: uses getSCSize function included from twDriver.c
 * *******************************************************************/
static UInt getAccountSize(Account *acct, Boolean enc)
{
    UInt length=0;
    length= sizeof(acct->SystemID)+sizeof(acct->AccountID)+
        StrLen(acct->username)+StrLen(acct->password)+
        StrLen(acct->type)+StrLen(acct->comment)+4;

        // if the account will be encrypted call getSCSize, 
        // otherwise just return the raw length
    if(enc)
        return getSCSize(length);
    else
        return length;
}

/********************************************************************
 * Function: addAccount
 * Description:  function responsible for writing a packed account record
 * to the database. This function is slightly different than addPassword
 * or addSystem. In order to increase search speed in the account 
 * database we will write the systemID of the account to the database
 * without encrypting it. not that this is the systemID that the program
 * uses internally, it has nothing to do with real user data.
 * ******************************************************************/
static void addAccount(UInt sysID, RecordBuffer acct, VoidHand acctDBrec)
{
    UInt length=0;
    CharPtr ch;
    UInt offset=0;
    Err err;
        
        // calculate the size
    length= MemPtrSize(acct)+sizeof(sysID);

        // resize the handle and write.
    if(MemHandleResize(acctDBrec, length)==0)
    {
        ch=MemHandleLock(acctDBrec);
        DmWrite(ch, offset,  &sysID, sizeof(sysID));
        offset+=sizeof(sysID);
        DmWrite(ch, offset,  acct, MemPtrSize(acct));
        MemHandleUnlock(acctDBrec);
    }
}

/*******************************************************************
 * Function: getUniqueSystemID
 * Description: return a unique system id that can be assigned to a
 * new system.
 * Note: SystemID 0 is reserved for the Unfiled system. Unfiled is 
 * where beamed accounts go when they are first received.
 * ****************************************************************/
static UInt getUniqueSystemID(void)
{
    UInt i, id;

        // get the number of system records. note that we cant just
        // use totalItems++ because systems can be added and deleted
        // at will... 
    UInt totalItems=DmNumRecordsInCategory(SystemDB, dmAllCategories);
    UInt max=0;

        // iterate through the records, and we will return the highest 
        // one it higher than the highest current value.
    for(i=0; i< totalItems; i++)
    {
        System sys;
        RecBuff scr;
        VoidPtr p;
        if(scr=MemHandleNew(1))
        {
            id=getSIDFromSystemIndex(i,scr);
            if(id>max)
            {
                max=id;
            }
            p=MemHandleLock(scr);
            MemSet(p, MemPtrSize(p), 0);
            MemHandleUnlock(scr);
            MemHandleFree(scr);
        }
    }
    return max+1;
} 
    
/*******************************************************************
 * Function: getUniqueAccountID
 * Description: return a unique account id that can be assigned to a
 * new account.
 * ****************************************************************/
static UInt getUniqueAccountID(void)
{
    UInt i, id;

        // get the number of account records. note that we cant just
        // use totalItems++ because account can be added and deleted and
        // they are in alphabetical sort order, not is AccountID order.
    UInt totalItems=DmNumRecordsInCategory(AccountDB, dmAllCategories);
    UInt max=0;

        // iterate through the records, and we will return the highest 
        // one it higher than the highest current value.
    for(i=0; i< totalItems; i++)
    {
        Account acc;
        id=getSIDFromAccountIndex(i);
        if(id>max)
        {
            max=id;
        }
    }
    return max+1;
} 
    
/*************************************************************************
 * Function: getIndexOfNthAcct
 * Description: the nth account in a system is not necessarily the nth 
 * account in the database, (remember sort order). this function will
 * take n as the nth account in a system and return the records index 
 * in the database.
 * **********************************************************************/
static UInt getIndexOfNthAcct(UInt n)
{
    UInt i, id;
    UInt totalItems=DmNumRecordsInCategory(AccountDB, dmAllCategories);
    UInt index=0;
    UInt counter=0;
    UInt recNum=0;
    Boolean cont=true;

        // iterate through the accounts, till we find the nth account for
        // the current systemID.
    for(i=0; (i< totalItems) && cont; i++)
    {
        Account ac;
                // note: getSIDFromAccountIndex is fast because it doesnt need 
                // to decrypt anything.
        id=getSIDFromAccountIndex(i);
        if(id==currentSID)
        {
            counter++;
            index=i;
            if(counter>n)   
                cont=false;
        }
    }
    return index;
} 

/***************************************************************************
 * Function: encryptPlainAccounts
 * Description: if an account is beamed from one palm to anoter and the 
 * receiving palm does not have strip as the currently running program
 * the record is written to the database in cleartext, and it is flagged
 * for encryption the next time strip is run. This function searches for these
 * flagged accounts, it encrypts them.
 * *************************************************************************/
static void encryptPlainAccounts()
{
    UInt totalAItems=DmNumRecordsInCategory(AccountDB, dmAllCategories);
    UInt i, index=0;
    Int id;

        // loop through the records in the database and see if they are flagged.
    for(i=0; i< totalAItems; i++)
    {
        id=getSIDFromAccountIndex(i);
        //if the flag is set (SystemID == -1) then we encrypt the record.
        if(id==-1)
        {
            Account ac;
            RecBuff rH=DmGetRecord(AccountDB, i);
            if(rH)
            {
                RecordBuffer pac, scratch, scratch2;
    
                pac=MemHandleLock(rH);
                if(scratch=MemPtrNew(MemHandleSize(rH)))
                {
            
                        // unpack the plain account.
                    UnpackAccount(&ac, pac, scratch, SysPass, MemHandleSize(rH), false, true);
            
                    if(scratch2=MemPtrNew(getAccountSize(&ac, true)))
                    {
                        id=0;
                        ac.SystemID=0;
                        ac.AccountID=getUniqueAccountID();
                
                            // find sort position (plain accounts are never sorted.
                        index=DmFindSortPosition(AccountDB, &ac, 0, (DmComparF *) SortPosAccountFunction, 0);

                            // pack it, encrypting this time, and add it.
                        PackAccount(scratch2, ac, SysPass, true); 
                        addAccount(id, scratch2, rH);
                        DmReleaseRecord(AccountDB, i, true); 

                            // move the record into sort order.
                        DmMoveRecord(AccountDB, i, index);
                
                        if(index>i)
                            i-=1;
                        
                        MemSet(scratch2, MemPtrSize(scratch2), 0);              
                        MemPtrFree(scratch2);
                    }
                    MemSet(scratch, MemPtrSize(scratch), 0);
                    MemPtrFree(scratch);
                }
                MemPtrUnlock(pac);
             }
        }
    }
}   

/***************************************************************************
 * Function: changePassword
 * Description: handles changing the system password based upon the 
 * password change screen. Basically checks that current password is correct,
 * checks that the new password was entered correctly, then re-encrypts the
 * databases based upon the new password.
 * ************************************************************************/
static void changePassword(void)
{
    char *oP, *pWS, *cP;

        // compact text fields. this is not really neccessary, but it is
        // a good habit as it frees wasted memory.
    FldCompactText(GetObjectFromActiveForm(ChangePasswordCurrentField));
    FldCompactText(GetObjectFromActiveForm(ChangePasswordNewField));
    FldCompactText(GetObjectFromActiveForm(ChangePasswordField));
    
        // get pointers to all the fields.
    cP=FldGetTextPtr(GetObjectFromActiveForm(ChangePasswordNewField));
    oP=FldGetTextPtr(GetObjectFromActiveForm(ChangePasswordCurrentField));  
    pWS=FldGetTextPtr(GetObjectFromActiveForm(ChangePasswordField));

        // if the user doesnt enter anything, just assume they dont really want to
        // change the password and pop them back to the last form they were at.
    if((!cP)||(!oP)||(!pWS)||(!StrLen(cP))||(!StrLen(oP))||(!StrLen(pWS)))
    {
        changeForm(oldForm);
    }

        // if they specified the wrong system password they arent allowed to change 
        // the system passwords.
    else if(StrCompare(oP, SysPass))
    {
        FrmCustomAlert(GenericError, "Incorrect System Password!", NULL, NULL);
    }
        
        // verification failed, warn the user.
    else if(StrCompare(pWS, cP))
    {
        FrmCustomAlert(GenericError, "New passwords do not match!", NULL, NULL);
    }

        // everything looks good, lets go to work.
    else 
    {
            // total number of records to re-write
        UInt totalAItems=DmNumRecordsInCategory(AccountDB, dmAllCategories);
        UInt totalSItems=DmNumRecordsInCategory(SystemDB, dmAllCategories);
        RecordBuffer pac=NULL, scratch=NULL, scratch2=NULL;
        UInt i=0, index=0;
        VoidHand rH;

            // re-encrypt the password 
        if(rH=DmGetRecord(PasswordDB, 0))
        {   
            if(scratch=MemPtrNew(24))
            {
                PackPassword(scratch, pWS);
                addPassword(scratch, rH); 
                MemSet( scratch, MemPtrSize(scratch), 0);           
                MemPtrFree(scratch);
            }
            DmReleaseRecord(PasswordDB, 0, true); 
        }
      
            // loop through the systems and re-encrypt
        for(i=0; i< totalSItems; i++)
        {
            System sys;
            if(rH=DmGetRecord(SystemDB, i))
            {
                pac=MemHandleLock(rH);
                if(scratch=MemPtrNew(MemPtrSize(pac)))
                {
                        // decrypt the system with old password
                    UnpackSystem(&sys, pac, scratch, SysPass, MemHandleSize(rH), true);
                    if(scratch2=MemPtrNew(getSystemSize(&sys)))
                    {
                            // re-encrypt with new password
                        PackSystem(scratch2, sys, pWS, true); 
                        addSystem(scratch2, rH);
                        MemSet(scratch2, MemPtrSize(scratch2), 0);
                        MemPtrFree(scratch2);
                    }
                    MemSet(scratch, MemPtrSize(scratch), 0);
                    MemPtrFree(scratch);
                }
                MemHandleUnlock(rH);
                DmReleaseRecord(SystemDB, i, true);
            }
        }
        
            // loop through the accounts and re-encrypt
        for(i=0; i< totalAItems; i++)
        {
            UInt id;
            Account ac;
            if(rH=DmGetRecord(AccountDB, i))
            {
                pac=MemHandleLock(rH);
                if(scratch=MemPtrNew(MemPtrSize(pac)))
                {
                
                        // decrypt the system with old password
                    UnpackAccount(&ac, pac, scratch, SysPass, MemHandleSize(rH), true, true);
                    id=ac.SystemID;
                    if(scratch2=MemPtrNew(getAccountSize(&ac, true)))
                    {
                            // re-encrypt with new password
                        PackAccount(scratch2, ac, pWS, true); 
                        addAccount(id, scratch2, rH);
                        MemSet(scratch2, MemPtrSize(scratch2),0); 
                        MemPtrFree(scratch2);
                    }
                    MemSet(scratch, MemPtrSize(scratch), 0);
                    MemPtrFree(scratch);
                }
                MemHandleUnlock(rH);
                DmReleaseRecord(AccountDB, i, true);
            }
        }
        
            // free the old password memory and copy the new password.
        if(SysPass)
        {
            MemSet(SysPass, MemPtrSize(SysPass), 0);
            MemPtrFree(SysPass);
        }
        if(SysPass=MemPtrNew(StrLen(pWS)+1))
            StrCopy(SysPass, pWS);
        
        changeForm(StripSystemForm);
    }
}

/*************************************************************************
 * Function: initSystems
 * Description: Simple function creates the "Unfiled category" with 
 * SystemID of 0
 * **********************************************************************/
static void initSystems()
{
    System sys;
    UInt index=0;
    VoidHand rec;
    if(rec=DmNewRecord(SystemDB, &index, 1))
    { 
            // set up the system info
        sys.SystemID=0;
        sys.name="Unfiled\0";
    
        if(rec)
        {          
            RecordBuffer scratch;
            if(scratch=MemPtrNew(getSystemSize(&sys)))
            {
                    // encrypt and write it.
                PackSystem(scratch, sys, SysPass, true);
                addSystem(scratch, rec);
                MemSet(scratch, MemPtrSize(scratch), 0);
                MemPtrFree(scratch);
            }
        }
        DmReleaseRecord(SystemDB, index, true);
    }
}
    
/*********************************************************************
 * Function: initPassword
 * Description: pass it the string password that the user originally 
 * requests, and write it out to the database.
 * *******************************************************************/
static void initPassword(char * pass)
{
    UInt i;
    UInt index =0;
    VoidHand rec;
    if(rec= DmNewRecord(PasswordDB, &index, 1))
    {
        RecordBuffer scratch;
        if(scratch=MemPtrNew(24))
        {                   
            PackPassword(scratch, pass);
            addPassword(scratch, rec);
            MemSet(scratch, MemPtrSize(scratch), 0);
            MemPtrFree(scratch);
        }
        DmReleaseRecord(PasswordDB, index, true); 
    }         
}

/********************************************************************
 * Function: DrawCharsToFitWidth
 * Description: simple utility call that will draw the chars of string
 * s into the rectange bounds. This is the function we use to 
 * draw the lists to the screen.
 * ******************************************************************/
static void DrawCharsToFitWidth(char *s, RectanglePtr r)
{
    SWord strLen=StrLen(s);
    SWord pixelWidth= r->extent.x;
    Boolean truncate;

    FntCharsInWidth(s, &pixelWidth, &strLen, &truncate);
    WinDrawChars(s, strLen, r->topLeft.x, r->topLeft.y);
}

/********************************************************************
 * Function: AccountListDrawFunction
 * Description: Callback function that does the drawing of the 
 * account names to the accountList
 * ******************************************************************/
static void AccountListDrawFunction(UInt itemNum, RectanglePtr bounds, CharPtr *data)
{
    UInt index=0, seekAmount=itemNum, length;
    VoidHand rec;
    Account ac;
    RecBuff scr;
    RecordBuffer toDraw;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    
    if(scr=MemHandleNew(1))
    {
            // get the account indicated by itemNum
        getAccountFromIndex(getIndexOfNthAcct(itemNum), scr, &ac);
    
            // if there is a account type, cat it onto the end
        length=StrLen(ac.username)+StrLen(ac.type)+5;
        if(toDraw=MemPtrNew(length))
        {
            StrCopy(toDraw, ac.username);
            if(StrLen(ac.type))
            {
                StrCat(toDraw, " - ");
                StrCat(toDraw, ac.type);
            }

                // draw it
            DrawCharsToFitWidth((CharPtr) toDraw, bounds);
            MemSet(toDraw, MemPtrSize(toDraw), 0);
            MemPtrFree(toDraw);
        }
        MemHandleFree(scr);    
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
}

/*****************************************************************
 * Function: numAccountsInSystem
 * Description: count the number of accounts for a given system 
 * ID 
 * ***************************************************************/
static UInt numAccountsInSystem(UInt id)
{
    UInt i, acid;
    UInt totalItems=DmNumRecordsInCategory(AccountDB, dmAllCategories);
    UInt numAccounts=0;
    UInt recNum=0;
    
        // iterate through the records, increment when we find a match
    for(i=0; i< totalItems; i++)
    {
        Account ac;
        acid=getSIDFromAccountIndex(i);
        if(acid==id)
            numAccounts++;
    }
    return numAccounts;
} 

/********************************************************************
 * Function: SystemListDrawFunction
 * Description: Callback function that does the drawing of the 
 * system names to the SystemList
 * ******************************************************************/
static void SystemListDrawFunction(UInt itemNum, RectanglePtr bounds, CharPtr *data)
{
    Int seekAmount=itemNum;
    UInt index=0;
    System sys;
    RecBuff scr;

#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    if(scr=MemHandleNew(1))
    {
        VoidPtr p;
            // get the system and draw it
        getSystemFromIndex(itemNum, scr, &sys);
        DrawCharsToFitWidth(sys.name, bounds);
         
        p=MemHandleLock(scr);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(scr);
        MemHandleFree(scr);
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
}

    
/*********************************************************************
 * Function: AccountFormInit
 * Description: function will set up the list to be drawn
 * find out how many items will be in the list, and install the
 * AccountListDrawFunction
 * *******************************************************************/
static void AccountFormInit(void)
{

    UInt numAcc=numAccountsInSystem(currentSID);
    ListPtr list =GetObjectFromActiveForm(accountList); 
    LstSetListChoices(list, NULL,numAcc);
    LstSetDrawFunction(list, AccountListDrawFunction); 
    if(selectedAccount>0)
    {
        LstSetSelection(list, selectedAccount);
        LstMakeItemVisible(list,selectedAccount);
    }
}

/*********************************************************************
 * Function: SystemFormInit
 * Description: function will set up the list to be drawn
 * find out how many items will be in the list, and install the
 * SystemListDrawFunction
 * *******************************************************************/
static void SystemFormInit(void)
{
    UInt numSys=DmNumRecordsInCategory(SystemDB, dmAllCategories);
    ListPtr list =GetObjectFromActiveForm(SystemList); 
    LstSetListChoices(list, NULL, numSys);
    LstSetDrawFunction(list, SystemListDrawFunction); 
    if(selectedSystem>0)
    {
        LstSetSelection(list, selectedSystem);
        LstMakeItemVisible(list,selectedSystem);
    }
}

/**********************************************************************
 * Function: DisplayInfoBox
 * Description: pops up a dialog box informing the user of how many
 * systems and accounts that strip is tracking.
 * ********************************************************************/
static Word DisplayInfoBox(void)
{
    char s[6], a[6];
    StrIToA(s, DmNumRecordsInCategory(SystemDB, dmAllCategories));
    StrIToA(a, DmNumRecordsInCategory(AccountDB, dmAllCategories));

    FrmCustomAlert(infoDialog, s, a, NULL);
    
}

/***********************************************************************
 * Function: DisplayAboutBox
 * Description: pops up the modal dialog "about strip"
 * *********************************************************************/
static Word DisplayAboutBox(void)
{
    FormPtr previousForm=FrmGetActiveForm();
    FormPtr frm = FrmInitForm(aboutAlert);
    Word button;
    
        // pop up the dialog.
    button=FrmDoDialog(frm);

        // set the previous form.
    if(previousForm)
        FrmSetActiveForm(previousForm);

        // Must call FrmDeleteForm or the memory never get released.
    FrmDeleteForm(frm);
    return button;
}

/**************************************************************************
 * Function: SelectSystemInit
 * Description: this function sets up the number of choices and the 
 * height of the popup list in the EditAccountForm screen. This list
 * and corresponding popup will be used to move accounts from one system
 * to another (especialy after receiving a beam
 * *************************************************************************/
static void SelectSystemInit(void)
{
    UInt numSys=DmNumRecordsInCategory(SystemDB, dmAllCategories);
    ListPtr list=GetObjectFromActiveForm(EditAccountPopupList);
    
        // if there are less than 8 systems, set the list height to be
        // the number of systems. If there are more than 8 systems a maximum
        // of 8 will be shown at a time (without scrolling).
    if(numSys<8)
        LstSetHeight(list, DmNumRecordsInCategory(SystemDB, dmAllCategories));
    else
        LstSetHeight(list, 8);

    LstSetListChoices(list, NULL, numSys);
}

/********************************************************************************
 * Function: StartApplication
 * Description: This is the first function that gets called and it is 
 * responsible for initializing databases and other global variables, checking
 * to see whether private records will be shown and calculating the auto-off time
 * for the current system.
 * ******************************************************************************/
static Err StartApplication(void)
{
    UInt mode=dmModeReadWrite;
    Err errors=0;
    Boolean created;

        // set current form to be the password opener
    currentForm= PasswordForm; 

        // check about private records
    hideSecretRecords=PrefGetPreference(prefHidePrivateRecords);
    if(!hideSecretRecords)
        mode |= dmModeShowSecret;


        // open or create databases.
    errors=getDatabase(&SystemDB, systemDBType, StripCreator, mode, 0, systemDBName, &created); 
    errors=getDatabase(&AccountDB, accountDBType, StripCreator, mode, 0, accountDBName, &created); 
    errors=getDatabase(&PasswordDB, passwordDBType, StripCreator, mode, 0, passwordDBName, &created); 
    
        // if the password database does not exist, or there are no records in it note that 
        // this is the first run, so the user will be prompted to enter a new password.
    if(created||(DmNumRecordsInCategory(PasswordDB, dmAllCategories)==0))
    {
        firstRun=true;
    }
    
        // set up the timer stuff.  start by setting auto off time to 0. the function returns the old 
        // auto off time. If the oldAutoOffTime is 0 then the system will never shut down. this is not
        // the behavior that we want, so we reset the autoOffTime to 300. if the oldAutoOffTime is not 
        // 0 then we set it back immediatly.  Note that in the StopApplication function we 
        // set the autoOffTime back to what it was before this program started no matter what.
    oldAutoOffTime=SysSetAutoOffTime(0);
    if(oldAutoOffTime==0)
        autoOffTime=300;
    else
        autoOffTime=oldAutoOffTime;
    SysSetAutoOffTime(autoOffTime);
        
        // initialize the lastEventTime variable.
    lastEventTime=TimGetSeconds();
    
    return 0;
}

/************************************************************************
 * Function: StopApplication
 * Description:  this function is responsible for stopping all application
 * related activity, freeing used memory, resetting the auto-off time, and
 * closing the databases.
 * **********************************************************************/
static void StopApplication(void)
{
        // re-set the autoOffTime.
    SysSetAutoOffTime(oldAutoOffTime);

        // free the System, password and account caches
    freeCache();
    FrmCloseAllForms();

        // close databases.
    DmCloseDatabase(SystemDB);
    DmCloseDatabase(AccountDB);
    DmCloseDatabase(PasswordDB);
}

/*****************************************************************************
 * Function: changeForm
 * Description: utility function that changes the current form and tracks
 * the current and last form shown.
 * ***************************************************************************/
static void changeForm(Word id)
{
    oldForm=currentForm;
        
        // this function changes the current form and sends a form load event.
    FrmGotoForm(id);
    currentForm=id;
}

/********************************************************************************
 * Function: checkPassword
 * Description: this is the function responsible for checking the 
 * input password value of the authentication form. 
 * ******************************************************************************/
static void checkPassword(void)
{
    RecordBuffer pass1, scratch=NULL;
    char * input;   
    UInt recNum=0, index=0, len, len2;
    VoidHand rec;
    mdKey in, saved;
    
        // compact text and get a pointer.
    FldCompactText(GetObjectFromActiveForm(PasswordField));
    input=FldGetTextPtr(GetObjectFromActiveForm(PasswordField));
    
        // if SysPass is defined, free it. this happens when Strip locks
        // itself after the timeout.
    if(SysPass)
    {
        MemSet(SysPass, MemPtrSize(SysPass), 0);
        MemPtrFree(SysPass);
    }
        
        // if its the first time the user has used the program we need 
        // to set some things up. 
    if(firstRun)
    {
            // if they entered a password
        if(input&&StrLen(input))
        {
                // store the new password to  db
            initPassword(input);           
        }
            
            // otherwise tell them that the password they enter will be the default password
        else
        {
            FrmCustomAlert(GenericError, "This is the first time you are using Strip! The password you enter at the prompt will become the default system password.", NULL, NULL);
            input=NULL;
            firstRun=true;
            return;
        }
    }
    
    if(input&&StrLen(input))
    {
            // read the password from the database, decrypt with the input text.
        if(rec=DmQueryRecord(PasswordDB,index)) 
        {
            pass1 = MemHandleLock(rec);
            if(scratch=MemPtrNew(24))
                UnpackPassword(pass1, scratch, input);
            MemHandleUnlock(rec);
        }
                   
            // the message digest of the password they provided should be exactly the
            // same as the message digest that was just decrypted out of the password
            // database. Do a MemCmp to make sure they are the same.
        MDString(input, in);
    
        if((!MemCmp(in, scratch, 16))&&input)
        {
                // if so, copy the password onto the system-password
            if(SysPass=MemPtrNew(StrLen(input)+1))
                StrCopy(SysPass, input);
            
                // if its the first run initialize the Unfiled system
            if(firstRun)
            {
               initSystems();
               firstRun=false;
            }
   
                // always scan the database for unencrypted accounts that have been
                // beamed over.  We dont want these plaintext accounts to 
                // hang arount to long.
            encryptPlainAccounts();

            if(scratch)
            {
                MemSet(scratch, MemPtrSize(scratch), 0);
                MemPtrFree(scratch);
            }
            
            MemSet(in, sizeof(in), 0);
            MemSet(saved, sizeof(saved), 0);
                
                // turn the underlining off on the password input fields. this is a bug-fix for
                // a redraw problem.
            SetAttributes(1,0,0,true,PasswordField);
            SetAttributes(1,0,0,true,CoverPasswordField);

                // SUCCESS!!!!!
            changeForm(StripSystemForm);
        }
        else
        {
                // FAILURE!!!!!!
                // free the memory and tell the user they entered the wrong password.
            FieldPtr fld=GetObjectFromActiveForm(PasswordField);
            FrmCustomAlert(GenericError, "The password you entered is incorrect", NULL, NULL);
            FldSetSelection(fld, 0, FldGetTextLength(fld));
            FldSetSelection(GetObjectFromActiveForm(CoverPasswordField), 0, FldGetTextLength(fld));
            
            if(scratch)
            {
               MemSet(scratch, MemPtrSize(scratch), 0);
               MemPtrFree(scratch);
               SysPass=NULL;
            }
        }
    }
        // null string is always wrong!!!
    else
        FrmCustomAlert(GenericError, "The password you entered is incorrect", NULL, NULL);    
}

/*******************************************************************
 * Function: editSystem
 * Description: this function is responsible for editing a system
 * record. It pops up a modal dialog which allows the user to 
 * modify the system name.
 * *****************************************************************/
static void editSystem(void)
{
        // get pointer to previous form and initalize the modal dialog
    FormPtr previousForm=FrmGetActiveForm();
    FormPtr frm = FrmInitForm(EditSystemForm);
    FieldPtr fld;
    UInt fldLen, txtLen;
    Word button;
    System sys;
    VoidHand rec;
    UInt recNum=selectedSystem, index2=0; 
    

        // draw the form
    FrmSetActiveForm(frm);
    FrmDrawForm(frm);

        // set event handler
    FrmSetEventHandler(frm, EditSystemHandleEvent);

        // write the system name to the field
    WriteEditableSystemInfoToScreen();
   
        // handle the button click
    button=FrmDoDialog(frm);

    if(button==editSystemOk)
    {  
            // user accepted
        sys.SystemID=currentSID;

            // compact text and get string pointer
        FldCompactText(GetObjectFromActiveForm(EditSystemName));
        sys.name=FldGetTextPtr(GetObjectFromActiveForm(EditSystemName));
        
        if(sys.name&&StrLen(sys.name))
        {
                // its a vaild name so we find the new sort position, and 
                // re-write the record
            index2=DmFindSortPosition(SystemDB, &sys, 0, (DmComparF *) SortPosSystemFunction, 0);
            if(rec=DmGetRecord(SystemDB,recNum)) 
            {
                RecordBuffer scratch;
                if(scratch=MemPtrNew(getSystemSize(&sys)))
                {
                    PackSystem(scratch, sys, SysPass, true);
                    addSystem(scratch, rec);
                    MemSet(scratch, MemPtrSize(scratch), 0);
                    MemPtrFree(scratch);
                }
                DmReleaseRecord(SystemDB, recNum, true);
            }

                // move the record to its new position
            DmMoveRecord(SystemDB, recNum, index2);
        }
    }
    else if(button==DeleteSystem)
    {
            // the want to delete the system
        deleteSystemFromDB();
    }
    
    FrmEraseForm(frm);
    FrmDeleteForm(frm);

    if(previousForm)
    {
            // re-set the old form
       FrmSetActiveForm(previousForm);
    } 
    changeForm(StripSystemForm);
}

/**********************************************************************
 * Function: editAccount
 * Description: this function is responsible for changing account 
 * information based upon the editited contents of the EditAccountForm
 * ********************************************************************/
static void editAccount(void)
{
    VoidHand rec;
    VoidPtr p;
    RecordBuffer scratch;

        // set up the popuptrigger list that allows the user
        // to change the system an accont is linked to.
    UInt index=getIndexOfNthAcct(selectedAccount), index2=0;
    ListPtr list=GetObjectFromActiveForm(EditAccountPopupList);

        // compact the text fields
    FldCompactText(GetObjectFromActiveForm(EditAccountUsername));
    FldCompactText(GetObjectFromActiveForm(EditAccountPassword));
    FldCompactText(GetObjectFromActiveForm(EditAccountType));

        // get the new system ID and username
    CurAcc.SystemID=getSIDForSystemIndex(LstGetSelection(list));
    CurAcc.username=FldGetTextPtr(GetObjectFromActiveForm(EditAccountUsername));

    if(!CurAcc.username||!StrLen(CurAcc.username))
    {
        FrmCustomAlert(GenericError, "You must enter a username!!", NULL, NULL);
        return;
    }

        // get the record to edit
    rec=DmGetRecord(AccountDB, index);

    CurAcc.password=FldGetTextPtr(GetObjectFromActiveForm(EditAccountPassword));
    CurAcc.type=FldGetTextPtr(GetObjectFromActiveForm(EditAccountType)); 

    if(!CurAcc.password)
        CurAcc.password=emptyString;

    if(!CurAcc.type)
    CurAcc.type=emptyString;

        // lock the comment
    CurAcc.comment=MemHandleLock(CurComment);

    index2=DmFindSortPosition(AccountDB, &CurAcc, 0, (DmComparF *) SortPosAccountFunction, 0);

        // write out the new account to the database
    if(scratch=MemPtrNew(getAccountSize(&CurAcc, true)))
    {
        PackAccount(scratch, CurAcc, SysPass, true);
        
        if( rec )
        {
            addAccount(CurAcc.SystemID, scratch, rec);
            DmReleaseRecord(AccountDB, index, true);
            DmMoveRecord(AccountDB, index, index2);
        }
        MemSet(scratch, MemPtrSize(scratch), 0);
        MemPtrFree(scratch);
    }
    
    if(CurComment)
    {
        p=MemHandleLock(CurComment);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(CurComment);
    }
    MemHandleFree(CurComment);
    changeForm(StripAccountForm);
}

/*************************************************************
 * Function: freeCurAcc
 * Description: function that resets the current account cache
 * this function will be called when the user cancels and
 * new or edit account reqest is cancelled.
 * Note: we must free the handle for the Current comment
 * ***********************************************************/
static void freeCurAcc(void)
{
    VoidPtr p;
    CurAcc.SystemID=0;
    CurAcc.AccountID=0;    
        
    CurAcc.username=NULL;
    CurAcc.password=NULL;

    if(CurComment)
    {
        p=MemHandleLock(CurComment);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(CurComment);  
        MemHandleFree(CurComment);
    }   

    CurAcc.comment=NULL;
}

/***********************************************************************
 * Function: createNewAccount
 * Description: this will get the information from the newAccountForm 
 * and creating and encrypting an account.
 * *********************************************************************/
static void createNewAccount(void)
{
    RecordBuffer scratch;
    VoidPtr p;
    VoidHand rec;
    UInt index =dmMaxRecordIndex;

        // get the systemID and a uniqe account id.
    CurAcc.SystemID=currentSID;
    CurAcc.AccountID=getUniqueAccountID();

        // compact text
    FldCompactText(GetObjectFromActiveForm(addAccountUsername));
    FldCompactText(GetObjectFromActiveForm(addAccountPassword));
    FldCompactText(GetObjectFromActiveForm(addAccountType));

    CurAcc.username=FldGetTextPtr(GetObjectFromActiveForm(addAccountUsername));

    if(!CurAcc.username||!StrLen(CurAcc.username))
    {
        FrmCustomAlert(GenericError, "You must enter a username!!", NULL, NULL);
        return;
    }

        // get the account type password and comment
    CurAcc.password=FldGetTextPtr(GetObjectFromActiveForm(addAccountPassword));
    CurAcc.type=FldGetTextPtr(GetObjectFromActiveForm(addAccountType));

    if(!CurAcc.password)
        CurAcc.password=emptyString;

    if(!CurAcc.type)
        CurAcc.type=emptyString;

    CurAcc.comment=MemHandleLock(CurComment);

        // get the sort position
    index=DmFindSortPosition(AccountDB, &CurAcc, 0, (DmComparF *) SortPosAccountFunction, 0);

    if(scratch=MemPtrNew(getAccountSize(&CurAcc, true)))
    {
        PackAccount(scratch, CurAcc, SysPass, true);

        if(rec = DmNewRecord(AccountDB, &index, 1))
        {
                // add the account
            addAccount(CurAcc.SystemID, scratch, rec);
            DmReleaseRecord(AccountDB, index, true);
        }
        MemSet(scratch, MemPtrSize(scratch), 0);
        MemPtrFree(scratch);
    }
    
    if(CurComment)
    {
        p=MemHandleLock(CurComment);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(CurComment);
    }
    
    MemHandleFree(CurComment);
    changeForm(StripAccountForm);
}

/************************************************************************************
 * Function: UnpackSystem
 * Description: This is a utility function that will take a packed System ,
 * optionally decrypt it based on the passed in password, and set up 
 * an unpacked system. 
 * **********************************************************************************/
static void UnpackSystem(System * sys, RecordBuffer  p, RecordBuffer  scratch, char * pass, UInt recLen, Boolean decrypt)
{
    PSystem * psys ;
    char *s;

        // if necessary, decrypt, otherwise just copy the memory to the scratch buffer
    if(decrypt)
        stripCrypt(SysPass, p, scratch, recLen , 0);
    else
        MemMove(p, scratch, recLen);

        // set up the system, pointing the name to the first char in the name string.
    psys=(PSystem *) scratch;
    s = psys->name;
    sys->SystemID=psys->SystemID;
    sys->name=s;
    s+=StrLen(s)+1;
}

/************************************************************************************
 * Function: PackSystem
 * Description: Utility function that takes an unpacked system, optionally encrypts
 * it and packs it into a buffer, strings are seperated by null characters. puts the 
 * content in retbuffer
 * *********************************************************************************/
static void PackSystem(RecordBuffer  retbuff, System sys, char * pass, Boolean encrypt)
{
    UInt offset=0;
    RecordBuffer psys;
    if(psys=MemPtrNew(MemPtrSize(retbuff)))
    {   
            // move the data into the buffer.
        MemMove(psys+offset, &sys.SystemID, sizeof(sys.SystemID));
        offset+=sizeof(sys.SystemID);
    
        MemMove(psys+offset,sys.name, StrLen(sys.name)+1);
        offset+=StrLen(sys.name)+1;
    
            //optionally decrypt it.
        if(encrypt)
            stripCrypt(pass, psys,retbuff, getSystemSize(&sys), 1);
        else
            MemMove(retbuff,psys, getSystemSize(&sys));
    
        MemSet(psys, MemPtrSize(psys), 0);
        MemPtrFree(psys);
    }
}

/************************************************************************************
 * Function: UnpackSIDFromSystem
 * Description: optionally decrypts a packed system and returns the system id for
 * that system
 * *********************************************************************************/
static UInt UnpackSIDFromSystem(RecordBuffer  p, RecordBuffer  scratch, char * pass, Boolean decrypt)
{
    PSystem * psys ;
    if(decrypt)
        stripCrypt(SysPass, p, scratch, BLOCK , 0);
    else
        MemMove(p, scratch, BLOCK);

    psys=(PSystem *) scratch;
    return psys->SystemID;
}

/***********************************************************************************
 * Function: UnpackPassword
 * Description: unpack a password and decrpt it using spass
 * *********************************************************************************/
static void UnpackPassword(RecordBuffer p, RecordBuffer scratch, char * spass)
{
    stripCrypt(spass, p, scratch, 24 , 0);
}

/***********************************************************************************
 * Function: PackPassword
 * Description: pack a password and encrypt it using spass
 * *********************************************************************************/
static void PackPassword(RecordBuffer retbuff, char * spass)
{                     
    mdKey pw;
    RecordBuffer ppass;
    if(ppass=MemPtrNew(24)) 
    {
        MDString(spass, pw);
        MemMove(ppass, pw, sizeof(pw));
        stripCrypt(spass, ppass, retbuff, 24, 1); 
        
        MemSet(ppass, MemPtrSize(ppass), 0);
        MemPtrFree(ppass);
    }
}

    
/************************************************************************************
 * Function: UnpackAccount
 * Description: This is a utility function that will take a packed account ,
 * optionally decrypt it based on the passed in password, and set up 
 * an unpacked account. isRec determines whether the packed account is a full record.
 * remember that fullrecords have the plaintext system id prepended, so if it
 * is a full record we will ignore this space.
 * **********************************************************************************/
static void UnpackAccount(Account * acct, RecordBuffer  p, RecordBuffer  scratch, char * pass, UInt recLen, Boolean decrypt, Boolean isRec)
{
    PAccount * pacct ;
    char *s;
    UInt offset=sizeof(offset);
    recLen=recLen-offset;
        
        // decrypt if neccessary
    if(decrypt)
        stripCrypt(pass, p+offset, scratch, recLen , 0);
    else
            // if buffer has the systemID header disregard it.
        if(isRec)
            MemMove(scratch, p+offset, recLen);
        else
            MemMove(scratch, p, recLen);

        // split record up into its different components.   
    pacct=(PAccount *) scratch;
    s = pacct->username;
    acct->SystemID=pacct->SystemID;
    acct->AccountID=pacct->AccountID;
    acct->username=s;
    s+=StrLen(s)+1;
    acct->password=s;
    s+=StrLen(s)+1;
    acct->type=s;
    s+=StrLen(s)+1;
    acct->comment=s;
    s+=StrLen(s)+1;
}

/************************************************************************************
 * Function: PackAccount
 * Description: Utility function that takes an unpacked account, optionally encrypts
 * it and packs it into a buffer, strings are seperated by null characters. puts the 
 * content in retbuffer
 * *********************************************************************************/
static void PackAccount(RecordBuffer retbuff, Account acct, char * pass, Boolean encrypt)
{
    UInt offset=0;
    RecordBuffer pacct;
    if(pacct=MemPtrNew(MemPtrSize(retbuff)))
    {  
            // pack the fields of the account buffer together
        MemMove(pacct+offset, &acct.SystemID, sizeof(acct.SystemID));
        offset+=sizeof(acct.SystemID);
        MemMove(pacct+offset, &acct.AccountID, sizeof(acct.AccountID));
        offset+=sizeof(acct.AccountID);
        MemMove(pacct+offset,acct.username, StrLen(acct.username)+1);
        offset+=StrLen(acct.username)+1;
        MemMove(pacct+offset,acct.password, StrLen(acct.password)+1);
        offset+=StrLen(acct.password)+1;
        MemMove(pacct+offset, acct.type, StrLen(acct.type)+1);
        offset+=StrLen(acct.type)+1;
        MemMove(pacct+offset,acct.comment, StrLen(acct.comment)+1);
        offset+=StrLen(acct.comment)+1;
    
            // optionally encrypt
        if(encrypt)
            stripCrypt(pass, pacct,retbuff, getAccountSize(&acct, true), 1);
        else
            MemMove(retbuff, pacct, getAccountSize(&acct, true));
    
        MemSet(pacct, MemPtrSize(pacct), 0);
        MemPtrFree(pacct);
    }
}

/***************************************************************************
 * Function: WriteEditableSystemInfoToScreen
 * Description: writes the current system name to the EditSystem modal
 * dialog.
 * ************************************************************************/
static void WriteEditableSystemInfoToScreen(void)
{
    UInt fldLen;
    FieldPtr fld;
    
        // set the string value
    setFieldFromString(EditSystemName, CurSys.name);

        // set the insertion point to be the end of the string
    fld=GetObjectFromActiveForm(EditSystemName);
    fldLen=FldGetTextLength(fld);
    FldSetInsertionPoint(fld, fldLen);
    FrmSetFocus(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(),EditSystemName));
}

/**************************************************************************
 * Function: HandleClickInPopup
 * Description: Handles the selection process of clicking the system list
 * popup in the edit accout form
 * ***********************************************************************/
static void HandleClickInPopup(EventPtr event)
{
    System s;
    VoidPtr p;
        // get pointers to the relevent structures
    ListPtr list=event->data.popSelect.listP;
    ControlPtr c=event->data.popSelect.controlP;
    UInt selection=event->data.popSelect.selection;
    RecBuff scratch;
    if(scratch=MemHandleNew(1))
    {
            // re-set the label to the new selection
        getSystemFromIndex(selection,scratch, &s);
        LstSetSelection(list, selection);
        CtlSetLabel(c, (CharPtr) s.name);
        
        p=MemHandleLock(scratch);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(scratch);
        MemHandleFree(scratch);
    }
}
    
/**************************************************************************
 * Function: getSIDFromSystemIndex
 * Description: returns the system id for the system located at a given
 * index.
 * ************************************************************************/
static UInt getSIDFromSystemIndex(UInt index, RecBuff tmp)
{
    UInt ret=0;
    VoidHand rec;

    rec=DmQueryRecord(SystemDB, index);
    if(rec)
    {
        RecordBuffer scratch, buff=MemHandleLock(rec);
            // resize the scratch handle
        if(MemHandleResize(tmp, BLOCK)==0)
        {
            scratch=MemHandleLock(tmp);
                // unpack the system ID.
            ret=UnpackSIDFromSystem(buff, scratch, SysPass, true);
        }
        MemHandleUnlock(rec);
    }
    return ret;
}

/**************************************************************************
 * Function: getSIDFromAccountIndex
 * Description: returns the system id for the system located at a given
 * index. Remember that the first few bytes of an account record are the
 * system ID of the account.  Just pop these off.
 * ************************************************************************/
static UInt getSIDFromAccountIndex(UInt index)
{
    UInt ret=0;
    VoidHand rec;

    rec=DmQueryRecord(AccountDB, index);
    if(rec)
    {
        RecordBuffer buff=MemHandleLock(rec);
        MemMove(&ret, buff, sizeof(ret));
        MemHandleUnlock(rec);
    }
    return ret;
}

/**************************************************************************
 * Function: getAccountFromIndex
 * Description: based upon an account index, this will locate the
 * account for that index, unpack and decrypt it and initialize acc
 * ************************************************************************/
static void getAccountFromIndex(UInt index, RecBuff tmp, Account * acc )
{
    UInt i=0;
    VoidHand rec;

        // query the record from the database
    rec=DmQueryRecord(AccountDB, index);
    if(rec)
    {
        RecordBuffer scratch, buff=MemHandleLock(rec);

            // resize buffer
        if(MemHandleResize(tmp, MemPtrSize(buff))==0)
        {
            scratch=MemHandleLock(tmp);

                // unpack the account
            UnpackAccount(acc, buff, scratch, SysPass, MemHandleSize(rec), true, true);
        }
        MemHandleUnlock(rec);
    }
}

/**************************************************************************
 * Function: getSystemFromIndex
 * Description: based upon a system database index, this will locate the
 * system for that index, unpack and decrypt it and initialize s
 * ************************************************************************/
static void getSystemFromIndex(UInt index, RecBuff tmp, System * s)
{
    UInt i=0;
    VoidHand rec;

        // query the record from the database
    rec=DmQueryRecord(SystemDB, index);
    if(rec)
    {
        RecordBuffer scratch, buff=MemHandleLock(rec);
        
            // resize the buffer
        if(MemHandleResize(tmp, MemPtrSize(buff))==0)
        {
            scratch=MemHandleLock(tmp);

                // unpack and decrypt the account
            UnpackSystem(s, buff, scratch, SysPass, MemHandleSize(rec), true);
        }
        MemHandleUnlock(rec);
    }
}

/****************************************************************************
 * Function: showCreateAccountForm
 * Description: sets up the newAccountForm, preps the CurComment handle 
 * and displays the newAccountForm
 * *************************************************************************/
static void showCreateAccountForm(void)
{
            // get a new handle
        if(CurComment=MemHandleNew(1))
        {
                // set the string to null
            StrCopy(MemHandleLock(CurComment), "\0");
            MemHandleUnlock(CurComment);
        }        
        changeForm(makeNewAccount);
}

/***************************************************************************
 * Function: showEditAccountForm
 * Description: sets up the EditAccountForm , preps the handle and 
 * displays the EditAccountForm
 * *************************************************************************/
static void showEditAccountForm(void)
{
        // if there is a comment, copy the comment into the new handle
    if(CurAcc.comment)
    {
        CharPtr tmp;
        if(CurComment=MemHandleNew(StrLen(CurAcc.comment)+1))
            StrCopy(MemHandleLock(CurComment), CurAcc.comment);
    }
        // otherwise set it to null string.
    else
    {
        if(CurComment=MemHandleNew(1))
            StrCopy(MemHandleLock(CurComment), "\0");
    }       

    MemHandleUnlock(CurComment);        
    changeForm(EditAccountForm);
}

/****************************************************************************
 * Function: WriteEditableAccountInfoToScreen
 * Description: write the current account info into the editable boxes
 * *************************************************************************/
static void WriteEditableAccountInfoToScreen(void)
{

    UInt fldLen;
    FieldPtr fld;
    ListPtr list;

        // get the popup, set the selection and label
    list=GetObjectFromActiveForm(EditAccountPopupList);
    LstSetDrawFunction(list, SystemListDrawFunction); 
    LstSetSelection(list, selectedSystem);
    CtlSetLabel(GetObjectFromActiveForm(EditAccountPopupTrigger), CurSys.name);

        // set up the fields 
    if(CurAcc.username)
        setFieldFromString(EditAccountUsername, CurAcc.username);
    if(CurAcc.password)
        setFieldFromString(EditAccountPassword, CurAcc.password);
    if(CurAcc.type)
        setFieldFromString(EditAccountType, CurAcc.type);
        
        // grab focus after the last character of the username field
    fld=GetObjectFromActiveForm(EditAccountUsername);
    fldLen=FldGetTextLength(fld);
    FldSetInsertionPoint(fld, fldLen);
    FrmSetFocus(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(),EditAccountUsername));
}

/*******************************************************************************
 * Function: editComment
 * Description: this will pop up the editable comment modal dialog that lets
 * the user modify the comment for an account
 * ****************************************************************************/
static void editComment(Word oldForm)
{
        // get previous form stuff
    FormPtr previousForm=FrmGetActiveForm();
    FormPtr frm = FrmInitForm(EditAccountCommentForm);
    VoidPtr tmp;
    FieldPtr fld;
    UInt fldLen, txtLen;

    FrmSetActiveForm(frm);
        
        // calling FrmDrawForm saves the pixels behind the modal dialog
        // so that the old form will be redrawn correctly when 
        // this modal dialog pops down.
    FrmDrawForm(frm);

        // set the event handler for scrolling and set the 
        // field attribute to indicate it has a linked scrollbar
        // so that it fires off field-changed events more often
    FrmSetEventHandler(frm, EditCommentHandleEvent);
    SetAttributes(1,1,1,true, EditAccountComment);
    
        // get the comment and set the handle.
    fld=GetObjectFromActiveForm(EditAccountComment);
    setFieldFromHandle(EditAccountComment, CurComment);
        
        // set the insertion point after the last character of the comment
    fldLen=FldGetTextLength(fld);
    FldSetInsertionPoint(fld, fldLen);
    FrmSetFocus(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(),EditAccountComment));
    UpdateScrollbar(EditAccountComment, EditAccountScrollbar);
    
        // pop it up
    FrmDoDialog(frm);

        // compact it.
    FldCompactText(fld);
        
        // setting the text handle to null directly will cause the handle we set earlier to 
        // NOT be deallocated automatically when the form pops down
    FldSetTextHandle(fld,NULL);
    
        // erase and delete the form
    FrmEraseForm(frm);
    FrmDeleteForm(frm);

        // re-set previous form as the active form
    if(previousForm)
    {
        FrmSetActiveForm(previousForm);
    }
}

/*******************************************************************************
 * Function: showComment
 * Description: this will pop up the non-editable comment modal dialog that lets
 * the user view the comment for an account
 * ****************************************************************************/
static void showComment()
{
    FormPtr previousForm=FrmGetActiveForm();
    FormPtr frm = FrmInitForm(EditAccountCommentForm);
    Word button;
    FieldPtr fld;

        // set the event handler for scrolling and set the 
        // field attribute to indicate it has a linked scrollbar
        // so that it fires off field-changed events more often.
        // also set the attribute to make sure the field is 
        // not editable
    FrmSetActiveForm(frm);
    FrmSetEventHandler(frm, EditCommentHandleEvent);
    SetAttributes(0,1,1,true, EditAccountComment);

        // calling FrmDrawForm saves the pixels behind the modal dialog
        // so that the old form will be redrawn correctly when 
        // this modal dialog pops down.
    FrmDrawForm(frm);

    if(CurAcc.comment)
        setFieldFromString(EditAccountComment, CurAcc.comment);
        
    UpdateScrollbar(EditAccountComment, EditAccountScrollbar);
    
    button=FrmDoDialog(frm);

        // erase and delete the form
    FrmEraseForm(frm);
    FrmDeleteForm(frm);

    if(previousForm)
    {
        FrmSetActiveForm(previousForm);
    }
}

/****************************************************************************
 * Function: WriteAccountInfoToScreen
 * Description: this function caches the selected account, 
 * and writes the accoutn info for the accout
 * *************************************************************************/
static void WriteAccountInfoToScreen(void)
{
    setFieldFromString(ShowAccountSystemName, CurSys.name);
        
        // cache the account
    cacheAccount(getIndexOfNthAcct(selectedAccount));

        // set up the fields
    if(CurAcc.username)
        setFieldFromString(AccountUsername, CurAcc.username);
    if(CurAcc.password)
        setFieldFromString(AccountPassword, CurAcc.password);
    if(CurAcc.type)
        setFieldFromString(AccountType, CurAcc.type);
}

/********************************************************************
 * Function: EditSystemHandleEvent
 * Description: Event handler for the edit system and new system 
 * modal dialogs. basically a wrapper to HandleCommonMenues
 * ******************************************************************/
static Boolean EditSystemHandleEvent(EventPtr event)
{
    Boolean     handled;
    CharPtr p;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;
    switch (event->eType)
    {
        case menuEvent:     
                // handle menus
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
        case frmCloseEvent:
            p=FldGetTextPtr(GetObjectFromActiveForm(EditSystemName));
            if(p)
                MemSet(p, StrLen(p), 0);
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/********************************************************************
 * Function: AddSystemHandleEvent
 * Description: Event handler for the edit system and new system 
 * modal dialogs. basically a wrapper to HandleCommonMenues
 * ******************************************************************/
static Boolean AddSystemHandleEvent(EventPtr event)
{
    Boolean     handled;
    CharPtr p;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;
    switch (event->eType)
    {
        case menuEvent:     
                // handle menus
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
            
        case frmCloseEvent:
            p=FldGetTextPtr(GetObjectFromActiveForm(addSystemName));
            if(p)
                MemSet(p, StrLen(p), 0);
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/**************************************************************************
 * Function: EditCommentHandleEvent
 * Description: event handler for the editComment screen. Specifically
 * handles the scrolling of the text
 * ***********************************************************************/
static Boolean EditCommentHandleEvent(EventPtr event)
{
    Boolean     handled;
    Word c;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;
    switch (event->eType)
    {
            // handle up and down button keys, and scroll the list
        case keyDownEvent:
            c=event->data.keyDown.chr;
            if(c==pageUpChr)
        PageScroll(up, EditAccountComment, EditAccountScrollbar);
            else if(c==pageDownChr)
        PageScroll(down, EditAccountComment, EditAccountScrollbar);
            break;

        case fldChangedEvent:
                // if the field height changes
            UpdateScrollbar(EditAccountComment, EditAccountScrollbar);
            handled=true;
            break;
            
        case sclRepeatEvent:
                // if the user hold down the scroll button..
            ScrollLines(event->data.sclRepeat.newValue-event->data.sclRepeat.value, false,
                    EditAccountComment, EditAccountScrollbar);
            break;

        case menuEvent:     
                // handle menus
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/**********************************************************************
 * Function: ChangePasswordHandleEvent
 * Description: this function handles the events for the changePassord
 * screen.
 * *******************************************************************/
static Boolean ChangePasswordHandleEvent(EventPtr event)
{
    Boolean     handled;
    CharPtr p;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;
    switch (event->eType)
    {
    
        case ctlSelectEvent:
            switch(event->data.ctlSelect.controlID)
            {
                    // if they accept the changes, sanity check them and accept
                case ChangePasswordOk:
                    changePassword();
                    break;

                    // just change to the last form if they cancel
                case ChangePasswordCancel:
                    changeForm(oldForm);
            }
            handled=true;
            break;

            // draw form and set focus in the first field
        case frmOpenEvent:  
            FrmDrawForm(FrmGetActiveForm());
            FrmSetFocus(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(),ChangePasswordCurrentField));
            handled = true;
            break;
            
            // handle menu events
        case menuEvent:     
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
        
        case frmCloseEvent:
            p=FldGetTextPtr(GetObjectFromActiveForm(ChangePasswordCurrentField));
            if(p)
                MemSet(p, StrLen(p), 0);
            p=FldGetTextPtr(GetObjectFromActiveForm(ChangePasswordNewField));
            if(p)
                MemSet(p, StrLen(p), 0);
            p=FldGetTextPtr(GetObjectFromActiveForm(ChangePasswordField));
            if(p)
                MemSet(p, StrLen(p), 0);        
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/****************************************************************************
 * Function: PasswordHandleEvent
 * Description: callback function that handles events for the form that
 * checks the password to login to the program. This one is a little tricky
 * . We accomplish an "echo-off" mode by covering the text the user writes
 * with a field contating '*' characters.  If you look at source file Strip.rcp
 * you will see that there are actually two fields that occupy the same exact
 * pixel locations on the field. which ever one is drawn last is the
 * one that will show up. If the user clicks the "echo-off" checkbox on 
 * the screen It will draw the CoverPasswordField over the actual password
 * field.  We therefore need to intecept any key down events that 
 * come into the event queue in order to handle the echo off stuff. This 
 * also poses a problem with selection. We need to make sure that the fields
 * both have the same selections and the same insertion points or it 
 * is confusing to the user. This function is called immediatly before
 * the default event handler, so it should never interfere with system
 * events.If anybody can think of a better way to allow
 * echo off, please let me know.
 * ************************************************************************/
static Boolean PasswordHandleEvent(EventPtr event)
{
    Boolean     handled;
    Word c, bak, startP, endP;
    CharPtr p;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;
    switch (event->eType)
    {
            // intercept key down events before the form gets them
        case keyDownEvent:
                // back up the character than comes it
            bak=c=event->data.keyDown.chr;

                // if its not an editing character, change the event character
                // to a '*'
            if(c != backspaceChr && c != leftArrowChr && c!= rightArrowChr)
            {
                event->data.keyDown.chr='*';
            }
                
                // force both fields to handle the event.
            FldHandleEvent(GetObjectFromActiveForm(CoverPasswordField),event);
            event->data.keyDown.chr=bak;
            FldHandleEvent(GetObjectFromActiveForm(PasswordField),event);

                // draw which ever field should be on top.
            if(passwordEchoOff)
                FldDrawField(GetObjectFromActiveForm(CoverPasswordField));
            else
                FldDrawField(GetObjectFromActiveForm(PasswordField));
                
            handled=true;
            break;
            
        case ctlSelectEvent:
            switch(event->data.ctlSelect.controlID)
            {

                    // they clicked ok, so check the password
                case PasswordOk:
                    checkPassword();
                    break;

                    // they want to change the echo status
                case EchoOffCheckbox:
                        // get the value of the checkbox
                    passwordEchoOff=CtlGetValue(GetObjectFromActiveForm(EchoOffCheckbox));
                        
                        // draw the appropriate field
                    if(passwordEchoOff)
                        FldDrawField(GetObjectFromActiveForm(CoverPasswordField));
                    else
                        FldDrawField(GetObjectFromActiveForm(PasswordField));
                    break;
            }
            handled=true;
            break;

        case frmOpenEvent:  
                // set the attributes so that the fields are underlined
            SetAttributes(1,1,0,true,PasswordField);
            SetAttributes(1,1,0,true,CoverPasswordField);

                // set the default password echo to be the last value of passwordEchoOff
            CtlSetValue(GetObjectFromActiveForm(EchoOffCheckbox),passwordEchoOff);
            FrmDrawForm(FrmGetActiveForm());
            FrmSetFocus(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(),PasswordField));
                
                // if its the first run, pop up a dialog letting them know that the password they choose will be
                // the one that they get
            if(firstRun)
                FrmCustomAlert(GenericError, "This is the first time you are using Strip! The password you enter at the prompt will become the default system password.", NULL, NULL);
            handled = true;
            break;
            
            // handle menu events
        case menuEvent:     
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
            
        case frmCloseEvent:
            p=FldGetTextPtr(GetObjectFromActiveForm(CoverPasswordField));
            if(p)
                MemSet(p, StrLen(p), 0);
            p=FldGetTextPtr(GetObjectFromActiveForm(PasswordField));
            if(p)
                MemSet(p, StrLen(p), 0);        
            break;
        
            // by default we will just check the bounds of selection for the password
            // field. If nothing is selected, then make sure both fields have the
            // same insertion point.
        default:
        {
            FieldPtr fld1=GetObjectFromActiveForm(PasswordField);
            FieldPtr fld2=GetObjectFromActiveForm(CoverPasswordField);
            FldGetSelection(fld1, &startP, &endP);

                // no selection
            if(startP==endP)
            {
                    startP=FldGetInsPtPosition(fld1);
                    FldSetInsertionPoint(fld2, startP);                
            }
            else
                FldSetSelection(fld2, startP, endP);
                
                // redraw appropriate field
            if(passwordEchoOff)
                        FldDrawField(GetObjectFromActiveForm(CoverPasswordField));
            else
                        FldDrawField(GetObjectFromActiveForm(PasswordField));
        }
        break;
            
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/*********************************************************************
 * Function: editAccountHandleEvent
 * Description: callback event handler for the edit account form
 * *******************************************************************/
static Boolean editAccountHandleEvent(EventPtr event)
{
    Boolean     handled;
    CharPtr p;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;
    switch (event->eType)
    {
    
        case ctlSelectEvent:
            switch(event->data.ctlSelect.controlID)
            {
                    // if they cancel, free the cache and change forms
                case editAccountCancel:
                    freeCurAcc();
                    changeForm(ShowAccount);
                    handled=true;
                    break;

                    // they accept the work, so commit it
                case editAccountOk:
                    editAccount();  
                    handled=true;
                    break;

                    // they want to edit the comment, so pop up the dialog
                case EditAccountCommentButton:
                    editComment(EditAccountForm);
                    handled=true;
                    break;
                    
                    // they want to delete it, so kiss it goodby
                case DeleteAccount:
                    deleteAccountFromDB();
                    handled=true;
                    break;

                    // they clicked the popup trigger, so call 
                    // the function that will draw the list.
                case EditAccountPopupTrigger:
                    SelectSystemInit();
                    break;
            }
            break;

            // they selected a new system, so handle the click in the popup
        case popSelectEvent:
            HandleClickInPopup(event);
            handled=true;
            break;

            // when it opens, draw the form and write out the info
        case frmOpenEvent:  
            FrmDrawForm(FrmGetActiveForm());
            WriteEditableAccountInfoToScreen();
            handled = true;
            break;
            
        case menuEvent:     
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
        
        case frmCloseEvent:
            p=FldGetTextPtr(GetObjectFromActiveForm(EditAccountUsername));
            if(p)
                MemSet(p, StrLen(p), 0);
            p=FldGetTextPtr(GetObjectFromActiveForm(EditAccountPassword));
            if(p)
                MemSet(p, StrLen(p), 0); 
            p=FldGetTextPtr(GetObjectFromActiveForm(EditAccountType));
            if(p)
                MemSet(p, StrLen(p), 0);        
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/********************************************************************
 * Function: showAccountHandleEvent
 * Description: callback event handler for the form that displays the
 * account information
 * *****************************************************************/
static Boolean showAccountHandleEvent(EventPtr event)
{
    Boolean     handled;
    CharPtr p;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;
    switch (event->eType)
    {
            // power stroke can cause this behavior if the 
            // user has their preferences set up to 
            // beam on power stroke. anyway- beam the account
        case keyDownEvent:
            if(event->data.keyDown.chr==sendDataChr)
                BeamAccount(getIndexOfNthAcct(selectedAccount));
            break;
            
        case ctlSelectEvent:
            switch(event->data.ctlSelect.controlID)
            {
                    // switch forms
                case AccountClose:
                    changeForm(StripAccountForm);
                    break;

                    // show this accouts comment
                case AccountCommentButton:
                    showComment();
                    break;

                    // edit this account
                case EditAccount:
                    showEditAccountForm();
                    break;

            }
            handled=true;
            break;

            // write the account info to the screen and then draw it.
        case frmOpenEvent:  
            FrmDrawForm(FrmGetActiveForm());
            WriteAccountInfoToScreen();
            handled = true;
            break;
            
        case menuEvent:     
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
        
         case frmCloseEvent:
            p=FldGetTextPtr(GetObjectFromActiveForm(ShowAccountSystemName));
            if(p)
                MemSet(p, StrLen(p), 0);
            p=FldGetTextPtr(GetObjectFromActiveForm(AccountUsername));
            if(p)
                MemSet(p, StrLen(p), 0); 
            p=FldGetTextPtr(GetObjectFromActiveForm(AccountPassword));
            if(p)
                MemSet(p, StrLen(p), 0); 
            p=FldGetTextPtr(GetObjectFromActiveForm(AccountType));
            if(p)
                MemSet(p, StrLen(p), 0);        
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/*********************************************************************
 * Function: newAccountHandleEvent
 * Description: event handler for the new account creation screen
 * *******************************************************************/
static Boolean newAccountHandleEvent(EventPtr event)
{
    Boolean     handled;
    CharPtr p;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;
    switch (event->eType)
    {
    
        case ctlSelectEvent:
            switch(event->data.ctlSelect.controlID)
            {
                    // they accept the info, so add the account
                case addAccountOk:
                    createNewAccount();
                    break;

                    // add a comment
                case AddAccountCommentButton:
                    editComment(makeNewAccount);
                    handled=true;
                    break;

                    // cancel, so free the cache and change back to 
                    // the previous form
                case addAccountCancel:
                    freeCurAcc();
                    changeForm(StripAccountForm);
                    break;
            }
            handled=true;
            break;

    
            // form open, draw the form and set the focus to the
            // first field
        case frmOpenEvent:  
            FrmDrawForm(FrmGetActiveForm());
            FrmSetFocus(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(),addAccountUsername));
            handled = true;
            break;
            
        case menuEvent:     
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
        
        case frmCloseEvent:
            p=FldGetTextPtr(GetObjectFromActiveForm(addAccountUsername));
            if(p)
                MemSet(p, StrLen(p), 0); 
            p=FldGetTextPtr(GetObjectFromActiveForm(addAccountPassword));
            if(p)
                MemSet(p, StrLen(p), 0); 
            p=FldGetTextPtr(GetObjectFromActiveForm(addAccountType));
            if(p)
                MemSet(p, StrLen(p), 0);        
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/********************************************************************
 * Function: createNewSystem
 * Description: this function will pop up the modal dialog and collect
 * the information to create a new system/category
 * ******************************************************************/
static void createNewSystem(void)
{
        // save the old form and initialize the new one
    FormPtr previousForm=FrmGetActiveForm();
    FormPtr frm = FrmInitForm(makeNewSystem);
    FieldPtr fld;
    UInt fldLen, txtLen;
    Word button;
    System sys;
    VoidHand rec;
    UInt index =dmMaxRecordIndex;

        // set active form, draw it
    FrmSetActiveForm(frm);
    FrmDrawForm(frm);

        // set the event handler and set the focus
    FrmSetEventHandler(frm, AddSystemHandleEvent);
    FrmSetFocus(frm, FrmGetObjectIndex(frm,addSystemName));
    
    button=FrmDoDialog(frm);
    if(button==addSystemOk)
    {  
            // if the user selected the ok button, add the system 
        sys.SystemID=getUniqueSystemID();
        FldCompactText(GetObjectFromActiveForm(addSystemName));
        sys.name=FldGetTextPtr(GetObjectFromActiveForm(addSystemName));

            // null strings are not valid
        if(sys.name&&StrLen(sys.name))
        {
                // find the sort position of the new system
            index=DmFindSortPosition(SystemDB, &sys, 0, (DmComparF *) SortPosSystemFunction, 0);
            if(rec = DmNewRecord(SystemDB, &index, 1))
            {
                RecordBuffer scratch;
                if(scratch=MemPtrNew(getSystemSize(&sys)))
                {
                        // pack it up and write it out
                    PackSystem(scratch, sys, SysPass, true);
                    addSystem(scratch, rec);
                    MemSet(scratch, MemPtrSize(scratch), 0);
                    MemPtrFree(scratch);
                }
                DmReleaseRecord(SystemDB, index, true);
            }
        }
    }
    
        // go back to the old form
    FrmEraseForm(frm);
    FrmDeleteForm(frm);

    if(previousForm)
    {
            // set active form and redraw the list
        FrmSetActiveForm(previousForm);
        SystemFormInit();
        LstDrawList(GetObjectFromActiveForm(SystemList));
    }
}

/*****************************************************************************
 * Function: deleteAccountFromDB
 * Description: this function will promt the user to delete and accout,
 * if they agree, it will delete the record from the database entirely
 * **************************************************************************/
static void deleteAccountFromDB(void)
{
        // prompt the user
    if(FrmCustomAlert(DeleteAlert, "Delete this account?", NULL, NULL)==0)
    {
            // delete it
        DmRemoveRecord(AccountDB, getIndexOfNthAcct(selectedAccount));
        changeForm(StripAccountForm);
    }
}

/******************************************************************************
 * Function: deleteSystemFromDB
 * Description: this function will prompt the user to delete a system and
 * all the associated account. it will then remove the system, then it
 * will iterate through the account database and delete all accounts with
 * that system id.
 * ****************************************************************************/
static void deleteSystemFromDB(void)
{
        // prompt the user
    if(FrmCustomAlert(DeleteAlert, "Delete this system and all associated accounts?", NULL, NULL)==0)
    {
        Account acct;
        UInt totalItems=DmNumRecordsInCategory(AccountDB, dmAllCategories);
        UInt i;
        Boolean delete=false;
        SDWord SysID=getSIDForSystemIndex(selectedSystem);

            // remove the system
        DmRemoveRecord(SystemDB, selectedSystem);

            // loop through the accounts
        for(i=0; (i< totalItems); i++)
        {
                // get the system id
            UInt id=getSIDFromAccountIndex(i);
            if(id==SysID) 
            {
                    // if the account needs to be deleted,
                    // mark it
                delete=true;
            }
            
            if(delete)
            {
                    // remove the record
                DmRemoveRecord(AccountDB, i);

                    // decriment both the current index (this
                    // will remain the same if we delete an account)
                    // and also the total items, so we dont end up 
                    // querying accounts that dont exist.
                i--;
                totalItems--;
            }
            delete=false;
        }
        changeForm(StripSystemForm);
    }
}   

/***********************************************************************
 * Function: getAIDForAccountIndex
 * Description: return the account id for an account by its index in
 * the database
 * *********************************************************************/
static UInt getAIDForAccountIndex(UInt n)
{
    Account ac;
    SDWord id;
    RecBuff scr;
    VoidPtr p;
    if(scr=MemHandleNew(1))
    {
        getAccountFromIndex(n, scr, &ac);
        id=ac.AccountID;
        p=MemHandleLock(scr);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(scr);  
        MemHandleFree(scr);
    }
    return id;
}       


/***********************************************************************
 * Function: showShowAccount
 * Description: get the current Account id and change the form
 * *********************************************************************/
static void showShowAccount(UInt index)
{
    currentAID=getAIDForAccountIndex(index);
    changeForm(ShowAccount);
}

/************************************************************************
 * Function: GetFocusObjectPtr
 * Description: return a pointer to whatever UI object currently has
 * the input focus
 * *********************************************************************/
static FieldPtr GetFocusObjectPtr(void)
{
    FormPtr frm;
    Word focus;
    FormObjectKind objType;

    frm=FrmGetActiveForm();
    focus=FrmGetFocus(frm);
        
        // if no focus return null
    if(focus==noFocus)
        return NULL;

    objType=FrmGetObjectType(frm, focus);

    if(objType == frmFieldObj)
        return (FrmGetObjectPtr(frm, focus));

        // handle tables, this version of strip currently doesnt have any tables
        // but who knows, it might someday
    else if(objType==frmTableObj)
        return (TblGetCurrentField (FrmGetObjectPtr(frm, focus)));

    return NULL;
}

/********************************************************************************
 * Function:BeamStream
 * Description: pass it a buffer, and the length in bytes of that buffer and 
 * it will beam it to another hand-held.  In this case the program will
 * always be running when we beam, so we are not terribly concerned with 
 * buffering
 * *****************************************************************************/
static Err BeamStream(ExgSocketPtr s, RecordBuffer buff, ULong bytes)
{
    Err err=0;

        // while there is no errors and we have bytes left
    while(!err && bytes>0)
    {
            // ExgSend does the actual beaming.
        ULong sent=ExgSend(s, buff, bytes, &err);
        bytes-=sent;

            // move the pointer along in the buffer
        buff=((char *) buff)+sent;
    }
    return err;
}

/***********************************************************************
 * Function: BeamAccount
 * Description: pass it the index of the account you want to beam 
 * and it will send that account and all associated information 
 * to another handlheld.  Accounts are not beamed in encrypted format,
 * because another handheld would have to know the key of the beaming
 * hand held, which is a security liability. As far as I know of, not 
 * too many people have access to infrared packet sniffers. and event if
 * they did the range of these devices would be line of sight, so im not
 * to worried.
 * ********************************************************************/
static void BeamAccount(UInt index)
{
    Account ac;
    Err err;
    ExgSocketType s;
    RecBuff scratch;
    RecordBuffer scratch2;
    VoidPtr p;

    if(scratch=MemHandleNew(1))
    {
            //decrypt an account
        getAccountFromIndex(index, scratch, &ac);
    
        if(scratch2=MemPtrNew(getAccountSize(&ac, true)))
        {
                // pack the account into a null seperated buffer,
                // but WITHOUT encrypting it
            PackAccount(scratch2, ac, SysPass, false); 
            
                // clean out the socket
            MemSet(&s, sizeof(s), 0);

                // make the description the username.
                // I am not using a mime-type so the 
                // extension is ".act" for the "file"
                // set the target to this creator-code "SJLO"
            s.description=ac.username;
            s.name="StripAccount.act";
            s.target=StripCreator;
        
                // initialize the socket and beam the buffer
            err=ExgPut(&s);
            if(!err)
                err= BeamStream(&s, scratch2, getAccountSize(&ac, true));
        
                // make sure to disconnect!!!
            err=ExgDisconnect(&s, err);
            
            MemSet(scratch2, MemPtrSize(scratch2), 0);
            MemPtrFree(scratch2);
        }
        p=MemHandleLock(scratch);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(scratch);
        MemHandleFree(scratch);
    }
}
    
/*************************************************************************
 * Function: HandleCommonMenus
 * Description: this is the event handler that handles all of the
 * menu events for the program
 * ***********************************************************************/
static Boolean HandleCommonMenus(Word menuID)
{
    FieldPtr fld;

    switch(menuID)
    {
            // edit commands.
        case EditUndo:
        case EditCut:
        case EditCopy:
        case EditPaste:
        case EditSelectAll:
            fld=GetFocusObjectPtr();
            if(!fld)
                return false;
            if(menuID==EditUndo)
                FldUndo(fld);
            else if(menuID==EditCut)
                FldCut(fld);
            else if(menuID==EditCopy)
                FldCopy(fld);
            else if(menuID==EditPaste)
                FldPaste(fld);
            else if(menuID==EditSelectAll)
                FldSetSelection(fld, 0, FldGetTextLength(fld));
            return true;

        case EditKeyboard:
                // pop up keyboard
            SysKeyboardDialog(kbdDefault);
            return true;
            
        case EditGrafitti:
                // pop up graffiti reference
            SysGraffitiReferenceDialog(referenceDefault);
            return true;

        case AboutMe:
                // pop up about box
            DisplayAboutBox();
            return true;

        case ChPass:
                // pop up change password form
            changeForm(ChangePasswordForm);
            return true;

        case NewSysPull:
                // pop up new system form
            createNewSystem();
            return true;

        case NewAccPull:
                // pop up new account form
            showCreateAccountForm();
            return true;

        case EditSysPull:
                // if the system id is not zero ("Unfiled"), pop up 
                // the edit system screen
            if(currentSID==0)
                FrmCustomAlert(GenericError, "You cannot edit the default system!!", NULL, NULL);
            else
                editSystem();
            return true;

        case EditAccPull:
                // pop up edit account screen
            showEditAccountForm();
            return true;

        case BeamAccPull:
                // beam the current account
            BeamAccount(getIndexOfNthAcct(selectedAccount));
            break;

        case AppInfo:
                // pop up app info box
            DisplayInfoBox();
            return true;

        default: 
                // not handled
            return false;
    }
}

/****************************************************************************
 * Function: AccountFormHandleEvent
 * Description: Event handler for the account form. the up and down buttons
 * will scroll the list one full page.  
 * *************************************************************************/
static Boolean AccountFormHandleEvent(EventPtr event)
{
    Boolean     handled;
    CharPtr p;
    Word c;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;


    switch (event->eType)
    {
            // handle up and down button keys, and scroll the list
        case keyDownEvent:
            c=event->data.keyDown.chr;
            if(c==pageUpChr)
                LstScrollList(GetObjectFromActiveForm(accountList), up, 7);
            else if(c==pageDownChr)
                LstScrollList(GetObjectFromActiveForm(accountList), down, 7);
            break;
            
        case ctlSelectEvent:
            switch(event->data.ctlSelect.controlID)
            {
                    // edit button allows the user to edit the selected system, if 
                    // its not SID 0 ("Unfiled" system)
                case EditSystem:
                    if(currentSID==0)
                        FrmCustomAlert(GenericError, "You cannot edit the default system!!", NULL, NULL);
                    else
                        editSystem(); 
                    break;

                    // done with this screen so go back to the system list
                case DoneAccount:
                    changeForm(StripSystemForm);    
                    break;

                    // create a new account
                case NewAccount:
                    showCreateAccountForm();
                    break;
            }
            handled=true;
            break;

            // if the user selects and account to view, pop up the
            // screen that will allow them to view it.
        case lstSelectEvent:
            selectedAccount=event->data.lstSelect.selection;
            showShowAccount(selectedAccount);   
            break;

            // write the system name to the screen, and draw the form
        case frmOpenEvent:  
            FrmDrawForm(FrmGetActiveForm());
            selectedAccount=0;
            setFieldFromString(AccountSystemName, CurSys.name);
            handled = true;
            break;
            
        case menuEvent:     
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
            
        case frmCloseEvent:
            p=FldGetTextPtr(GetObjectFromActiveForm(AccountSystemName));
            if(p)
                MemSet(p, StrLen(p), 0);       
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/******************************************************************************
 * Function: getSIDForSystemIndex
 * Description: this will get the system id for a given index in the
 * system database.
 * ****************************************************************************/
static UInt getSIDForSystemIndex(UInt i)
{
    SDWord id;
    System sys;
    RecBuff scratch;
    VoidPtr p;
    if(scratch=MemHandleNew(1))
    {
            // get the system for that index
        getSystemFromIndex(i, scratch,  &sys);
        id=sys.SystemID;
        
        p=MemHandleLock(scratch);
        MemSet(p, MemPtrSize(p), 0);
        MemHandleUnlock(scratch);
        MemHandleFree(scratch);
    }
    return id;
}       

/*********************************************************
 * Function: showAccountForm
 * Description: set up, cache and diplay the account form
 * ******************************************************/
static void showAccountForm(UInt index)
{
    UInt SysID;
    cacheSystem(index);

        // note the system ID
    currentSID=CurSys.SystemID; 
    changeForm(StripAccountForm);
}

/****************************************************************************
 * Function: SystemFormHandleEvent
 * Description: Event handler for the system form. the up and down buttons
 * will scroll the list one full page.  
 * *************************************************************************/
static Boolean SystemFormHandleEvent(EventPtr event)
{
    Boolean     handled;
    Word c;
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
    handled = false;


    switch (event->eType)
    {
            // handle up and down button keys, and scroll the list
        case keyDownEvent:
            c=event->data.keyDown.chr;
            if(c==pageUpChr)
                LstScrollList(GetObjectFromActiveForm(SystemList), up, 9);
            else if(c==pageDownChr)
                LstScrollList(GetObjectFromActiveForm(SystemList), down, 9);
            break;
            
        case ctlSelectEvent:
            switch(event->data.ctlSelect.controlID)
            {
                    // pop up new system modal dialog
                case NewSystem:
                    createNewSystem();
                    break;
                }
            handled=true;
            break;

            // the user selected a system in the list, so show the accouts
            // for that system
        case lstSelectEvent:
            selectedSystem=event->data.lstSelect.selection;
            showAccountForm(selectedSystem);    
            break;

            // draw the form
        case frmOpenEvent:  
            selectedSystem=0;
            FrmDrawForm(FrmGetActiveForm());
            handled = true;
            break;
            
        case menuEvent:     
            handled=HandleCommonMenus(event->data.menu.itemID);
            break;
    }
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
    return(handled);
}

/*********************************************************************
 * Function: ApplicationHandleEvent
 * Description: this is the event handler that we use to 
 * set up forms on frmLoadEvents. It sets the event handlers
 * for every other non modal form in the program
 * *******************************************************************/
static Boolean ApplicationHandleEvent(EventPtr event)
{
    FormPtr frm;
    Int     formId;
    Boolean handled = false;

    if (event->eType == frmLoadEvent)
    {
            // load and activate the requested form
        formId = event->data.frmLoad.formID;
        frm = FrmInitForm(formId);
        FrmSetActiveForm(frm);

            // set an event handler
        switch (formId)
        {
            case StripSystemForm:
                SystemFormInit();
                FrmSetEventHandler(frm, SystemFormHandleEvent);
                break;

            case StripAccountForm:
                AccountFormInit();
                FrmSetEventHandler(frm, AccountFormHandleEvent);
                break;
            
            case makeNewAccount:
                FrmSetEventHandler(frm, newAccountHandleEvent);
                break;

            case EditSystemForm:
                FrmSetEventHandler(frm, EditSystemHandleEvent);
                break;

            case EditAccountForm:
                FrmSetEventHandler(frm, editAccountHandleEvent);
                break;

            case ShowAccount:
                FrmSetEventHandler(frm, showAccountHandleEvent);
                break;
                
            case PasswordForm:
                FrmSetEventHandler(frm, PasswordHandleEvent);
                break;

            case ChangePasswordForm:
                FrmSetEventHandler(frm, ChangePasswordHandleEvent);
                break;

        }
        handled = true;
    }
    
    return handled;
}

/********************************************************************
 * Function: HandlePowerOff
 * Description: if the user presses the power button it locks the 
 * display and requires the user to re-enter the password 
 * when the unit turns on again
 * *****************************************************************/
static void HandlePowerOff(EventPtr event)
{
    if(event->eType==keyDownEvent)
        if(event->data.keyDown.chr==hardPowerChr)
            changeForm(PasswordForm);
                    
        // since this function pre-processes EVERY event, this 
        // is a good time to reset the last event timer we 
        // use to lock the screen when the unit sleeps
    lastEventTime=TimGetSeconds();
}

/**********************************************************************
 * Function: EventLoop
 * Description: this is the main event loop for the program. It 
 * listens for events for .5 seconds, if none come in, it checks 
 * wheter it should timeout and lock the program. If the unit
 * is scheduled to sleep within the next five seconds, it locks
 * the display. 
 * Note: timeout does not work properly if you overclock you 
 * palm's processor.
 * ********************************************************************/
static void EventLoop(void)
{
    EventType event;
    Word error;
        
        // get number of processor ticks that occur per second on
        // this device.
    
    do
    {
            // wait a bit for an event
        EvtGetEvent(&event, 50);
        if(event.eType!=nilEvent)
        {
                // pre-process event
            HandlePowerOff(&event);
            
                // first the system gets the event, then the Menu event handler, then the application
                // event handler, then finally the form event handler
            if (! SysHandleEvent(&event))
                if (! MenuHandleEvent(0, &event, &error))
                    if (! ApplicationHandleEvent(&event))
                        FrmDispatchEvent(&event);
        }
        else
        {
                // get the new time and check if the program should lock itself
            ULong newTime=TimGetSeconds();
            ULong time=lastEventTime+autoOffTime-5;
            if(time<=newTime)
                changeForm(PasswordForm);      
        }              
    }
    while (event.eType != appStopEvent);
}

/*************************************************************************
 * Function: ReadBytesIntoAccount
 * Description: this function is called when an account is
 * being beamed into the program. If Strip is running locally it
 * encrypts the account and writes it to the database. If Strip is 
 * not running the program does not know the password, so it writes
 * the account to the databse in plaintext format, and it will be
 * encrypted the next time Strip is run by the user
 * ***********************************************************************/
static Err ReadBytesIntoAccount(DmOpenRef db, ExgSocketPtr socket, ULong bytes, UInt aid, Boolean appRunning)
{
    char buffer[100];
    Err err;
    UInt index=0, i=0;
    ULong received;
    VoidHand rec, pac=NULL;
    RecordBuffer scrPtr, scratch, scratch2;
    ULong recSize=0;
    Boolean aRec=false;
    Account ac;

    do
    {
            // find out how many bytes to read
        Long bytesToRead=min(bytes, sizeof(buffer));

            // read these bytes int memory
        received=ExgReceive(socket, buffer, bytesToRead, &err);

        bytes-=received;

            // if there is no error, create a new memory handle
            // or resize the existing one to accomodate the new data
        if(!err)
        {
            if(!pac)
                pac=MemHandleNew(received);
            else
                MemHandleResize(pac, recSize+received);
            if(!pac)
            {
                err=1;
                break;
            }
        }
        aRec=true;
        scrPtr=MemHandleLock(pac);

            // pack the recieved data onto the buffer
        MemMove(scrPtr+recSize,buffer,received);
        MemHandleUnlock(pac);
        recSize+=received;
    }
    while(!err && received > 0 && bytes > 0 );
    
    scratch=MemPtrNew(recSize);
    scrPtr=MemHandleLock(pac);

        // unpack the beamed data into a local account
    UnpackAccount(&ac, scrPtr, scratch, NULL, recSize, false, false);

        // assign it a system ID of zero ("Unfiled" system)
    ac.SystemID=0;
    ac.AccountID=aid;
    
    scratch2=MemPtrNew(getAccountSize(&ac, true));

        // if the application is running, then encrypt it
        // otherwise just re-pack it.
    if(appRunning)
    {
        PackAccount(scratch2, ac, SysPass, true);
        index=DmFindSortPosition(db, &ac, 0, (DmComparF *) SortPosAccountFunction, 0);
    }
    else
        PackAccount(scratch2, ac, NULL, false);


        // create a new record and add hte accout to the database
    if(rec = DmNewRecord(db, &index, 1))
    {
        if(appRunning)
            addAccount(ac.SystemID, scratch2, rec);
        else
            addAccount(-1, scratch2, rec);

        DmReleaseRecord(db, index, true);
    }
    
        // free used memory
    MemSet(scratch, MemPtrSize(scratch), 0);
    MemSet(scratch2, MemPtrSize(scratch2), 0);
    MemSet(scrPtr, MemPtrSize(scrPtr), 0);
    
    MemPtrFree(scratch);
    MemPtrFree(scratch2);
    MemPtrFree(scrPtr);

    return err;
}
    
/*************************************************************************************
 * Function: RecieveBeamStream
 * Description: a wrapper function for ReadBytesIntoAccount. this function simply
 * gets a uniqe account ID or assigns the account id of 0. it accepts the
 * incoming socket, and disconnects when all the bytes have been read from the
 * stream
 * **********************************************************************************/
static Err RecieveBeamStream(DmOpenRef db, ExgSocketPtr socket, Boolean appRunning)
{
    Err err;
    UInt index;
    UInt aid;
    
    if(appRunning)
        aid=getUniqueAccountID();
    else
        aid=0;

        // accept incoming socket
    err=ExgAccept(socket);

        // read bytes into a new account
    if(!err)
        err=ReadBytesIntoAccount(db, socket, 0xffffffff, aid, appRunning);

        // dont forget to disconnect.
    err=ExgDisconnect(socket, err);

        // this application does not support the goto or find commands because 
        // of the encryption it uses. so dont go anywhere
    socket->goToCreator=NULL;
}

/**********************************************************************************
 * Function: PilotMain
 * Description: this is the function that is acctually called by the PalmOS when
 * application start and other events occur. we handle those system launch events
 * here
 * *******************************************************************************/
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    Err err = 0;        
    
        // request to start application normally
    if (cmd == sysAppLaunchCmdNormalLaunch)
    {
            // call StartApplication to initialize things, 
            // go to the opening form and enter the event loop,
            // until end.
        if ((err = StartApplication()) == 0)
        {
            FrmGotoForm(PasswordForm);
            EventLoop();
            StopApplication();
        }
    }
        // OS request to save data.
    else if(cmd == sysAppLaunchCmdSaveData)
    {
            // save all form data
        FrmSaveAllForms();
    }
        // OS Notification that a hotsync has occured. 
    else if(cmd == sysAppLaunchCmdSyncNotify)
    {
        DmOpenRef db;
        DWord romVersion;

            // get rom version, if it is greater than three, register
            // with the exchange manager.
        FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
        if(sysGetROMVerMajor(romVersion)>=3)
            ExgRegisterData(StripCreator, exgRegExtensionID, "act");
    }
        // OS notification that another palm pilot is attempting
        // to beam us data. 
    else if(cmd==sysAppLaunchCmdExgReceiveData)
    {
        DmOpenRef db;
        
        if(launchFlags & sysAppLaunchFlagSubCall)
        {
#ifdef __GNUC__
    CALLBACK_PROLOGUE
#endif
                // Our program is running and the database is already open
            db=AccountDB;
            FrmSaveAllForms();
                // recieve the beam
            err=RecieveBeamStream(db, (ExgSocketPtr) cmdPBP, true);
#ifdef __GNUC__
    CALLBACK_EPILOGUE
#endif
        }
        else
        {
                // our program is not running, so we open the database and 
                // then recieve the beam
            Boolean created;
            err=getDatabase(&db, accountDBType, StripCreator, dmModeReadWrite, 0, accountDBName, &created); 
            if(db)
                err=RecieveBeamStream(db, (ExgSocketPtr) cmdPBP, false);
            DmCloseDatabase(db);
        }
    }
    return err;
}

